#!/usr/bin/env python3

"""
ASN.1 encoding using numbered tags and length in bytes.
-1 indicates variable length.
"""
TYPES = {
   "NULL_DATA" :  (0, 0),
   "ARRAY"     :  (1, -1),
   "STRUCT"    :  (2, -1),
   "BOOL"      :  (3, 1),
   "BOOLSEQ"   :  (4, -1), # bit-string
   "INT32"     :  (5, 4),  # double-long
   "UINT32"    :  (6, 4),  # double-long-unsigned
                           # 7 is unused
                           # 8 is unused
   "OCTETS"    :  (9, -1), # Just a sequence of bytes
   "ASCII"     : (10, -1), # visible-string
                           # 11 is unused
   "UTF8"      : (12, -1), # characters encoded as utf8
   "BCD"       : (13, -1), # binary-coded decimal
                     # 14 is unused
   "INT8"      : (15, 1),
   "INT16"     : (16, 2),
   "UINT8"     : (17, 1),
   "UINT16"    : (18, 2),
   "COMP_ARR"  : (19, -1), # compact array
   "INT64"     : (20, 8),
   "UINT64"    : (21, 8),
   "ENUM"      : (22, 1),
   "F32"       : (23, 4),
   "F64"       : (24, 8),
   "DATETIME"  : (25, 12),
   "DATE"      : (26, 5),
   "TIME"      : (27, 4),
   "D_INT8"    : (28, 1), # Delta types
   "D_INT16"   : (29, 2),
   "D_INT32"   : (30, 4),
   "D_UINT8"   : (31, 1),
   "D_UINT16"  : (32, 2),
   "D_UINT32"  : (33, 4),

   "ARR_CONTENTS" : (129, -1) # Special array contents type tag, taken from the
                              # example on page 18 of draft 0.4 of the proposal
}


"""
Determine minimum number of bytes required to encode integer value
"""
def _min_bytes(value, signed=False):
    if value < 0:
        value = ~value
        signed = True

    max = 127 if signed else 255
    bytenum = 1
    while value > max:
        value = value >> 8
        bytenum += 1

    return bytenum, signed


def _encode_timestamp(timestamp, nullcoded=False, tagged=True):
    # Timestamps are encoded as octet strings.
    encoded_timestamp = b''
    if tagged:
        encoded_timestamp += _ber_encode_type("OCTETS")

    if not nullcoded:
        encoded_timestamp += (
            b'' +
            _ber_encode_length(12) + # length of the timestamp without typeflag
            timestamp.year.to_bytes(2, byteorder='big') +
            timestamp.month.to_bytes(1, byteorder='big') +
            timestamp.day.to_bytes(1, byteorder='big') +
            0x05.to_bytes(1, byteorder='big') + # Some kind of separator
            timestamp.hour.to_bytes(1, byteorder='big') +
            timestamp.minute.to_bytes(1, byteorder='big') +
            timestamp.second.to_bytes(1, byteorder='big') +
            b'\x00\x80\x00\x00' # Possibly more granular timestamp
        )
    else:
        # A length of 0 indicates NULL encoded octet string.
        encoded_timestamp += _ber_encode_length(0)
    return encoded_timestamp


def _ber_encode_element(typestring, element, length=None, tagged=True):
    if element is None:
        element = b''

    typelength = TYPES[typestring][1]
    if typelength < 0 and not length:
        length = len(element)
    elif typelength >= 0:
        if length:
            raise ValueError("Asked to encode fixed-length element with length tag")
        if len(element) != typelength:
            raise ValueError("Passed element of wrong length")

    enc_typetag = _ber_encode_type(typestring, length=length, tagged=tagged)

    return enc_typetag + element


def _ber_encode_type(typestring, length=None, tagged=True):
    encoded = b''
    if tagged:
        encoded += TYPES[typestring][0].to_bytes(1, byteorder='big')
    if length:
        encoded += _ber_encode_length(length)
    return encoded


def _ber_encode_length(length):
    if length < 0:
        raise ValueError("Something went horribly wrong, length is {}".format(length))

    if length < 128:
        return length.to_bytes(1, byteorder='big')
    else:
        bytenum, _ = _min_bytes(length)
        return ((0b10000000 | bytenum).to_bytes(1, byteorder='big') +
                length.to_bytes(bytenum, byteorder='big'))


def _int_to_bytes(value, bytelength=1, signed=False):
    return value.to_bytes(bytelength, byteorder='big', signed=signed)
#    try:
#        frac, value = math.modf(value)
#        b = int(value).to_bytes(length, byteorder='big', signed=signed)
#        if frac != 0:
#            raise TypeError("Unexpected fraction remainder!")
#    except OverflowError:
#        return None
#    except ValueError:
#        return None
#    return b


def _encode_common_inttype(value, length='min', signed=False):
    if length not in ['min', 8, 16, 32, 64]:
        raise ValueError("Invalid value passed for integer length: {}".format(length))

    bytenum, signed = _min_bytes(value)
    if signed:
        raise NotImplementedError("This code has not been written to take signed integers into account")

    if bytenum == 3:
        bytenum = 4
    elif bytenum > 4 and bytenum < 8:
        bytenum = 8
    elif bytenum > 8:
        raise ValueError("Element cannot be encoded as integer: {}".format(value))

    if length == 'min': # smallest integer possible
        length = bytenum * 8
    elif (length < bytenum * 8): # fixed integer requested that won't fit
        raise ValueError("Element cannot be encoded as requested {}-bit integer: {}".format(length, value))
    elif (length > bytenum * 8): # fixed integer requested, so set bytenum to the correct value
        bytenum = int(length / 8)

    return length, _int_to_bytes(value, bytelength=bytenum, signed=False)


def _encode_int(value, length='min', signed=False, tagged=True):
    if length not in ['min', 8, 16, 32, 64]:
        raise ValueError("Invalid value passed for integer length: {}".format(length))

    length, value_bytes = _encode_common_inttype(value, length=length, signed=signed)

    if length not in [8, 16, 32, 64]:
        raise ValueError("Invalid value returned for integer length: {}".format(length))

    typestring = "UINT" + str(length)
    return _ber_encode_element(typestring, value_bytes, tagged=tagged)


def _encode_delta(value, length='min', signed=False, tagged=True):
    if length not in ['min', 8, 16, 32]:
        raise ValueError("Invalid value passed for delta length: {}".format(length))

    length, value_bytes = _encode_common_inttype(value, length=length, signed=signed)

    if length not in [8, 16, 32]:
        raise ValueError("Invalid value returned for delta length: {}".format(length))

    typestring = "D_UINT" + str(length)
    return _ber_encode_element(typestring, value_bytes, tagged=tagged)


def _encode_measurement_struct_common(timestamp, nulldate=False, nullstatus=False, tagged=True):
    # Struct header, these structs have three elements
    if tagged:
        enc_header = _ber_encode_type("STRUCT", length=3)
    else:
        enc_header = b''

    # Timestamp
    enc_timestamp = _encode_timestamp(timestamp, nullcoded=nulldate, tagged=tagged)

    # Integer with value "0", reason for which is unclear but is most likely
    # the status byte mentioned in DSMR 5.
    # Examples in DLMS UA proposal 090 show this can be NULL coded.
    if nullstatus:
        enc_status = _ber_encode_element("NULL_DATA", None, tagged=tagged)
    else:
        enc_status = _ber_encode_element("UINT8", _int_to_bytes(0), tagged=tagged)

    return enc_header + enc_timestamp + enc_status


def _encode_measurement_struct(timestamp, measurement, nulldate=False, nullstatus=False, nullvalue=False, tagged=True):
    if measurement is None and not nullvalue:
        raise ValueError("Unable to NULL-encode measurement if not allowed")

    struct = _encode_measurement_struct_common(timestamp, nulldate=nulldate, nullstatus=nullstatus, tagged=tagged)

    # Measurement (actual meter reading)
    if measurement is None: # No further check here, see first line
        # NULL-encode the measurement
        enc_measurement = _ber_encode_element("NULL_DATA", None, tagged=tagged)
    else:
        # Encode the measurement as the smallest integer possible
        enc_measurement = _encode_int(measurement, tagged=tagged)
        if not enc_measurement:
            raise ValueError("Unable to encode integer")

    return struct + enc_measurement


def _encode_delta_struct(timestamp, delta, deltalength='min', nulldate=True, nullstatus=True, tagged=True):
    if deltalength == 'min' and not tagged:
        raise ValueError("Unable to combine untagged delta structs and min-length delta encoding")

    struct = _encode_measurement_struct_common(timestamp, nulldate=nulldate, nullstatus=nullstatus, tagged=tagged)

    # Encode the delta as requested
    enc_delta = _encode_delta(delta, length=deltalength, tagged=tagged)
    if not enc_delta:
        raise ValueError("Unable to encode delta")

    return struct + enc_delta


"""
The format of compact array encoding used here has been distilled from the
blue book, green book, but mostly from the example used in version 0.4 of
DLMS UA Contribution 090.

Compact array encoding always does NULL coding for dates, because it makes
no sense to save a few bytes on typing and then waste them on superfluous
information.

The only question is whether the *first* date in the array can be NULL coded
or not. For our proposed compact delta coding, the first date can be nulled.
"""
def _encode_compact_array(series, valuetype='D_UINT16', nullfirstdate=True, nullstatus=False):
    if valuetype not in ['D_UINT16']:
        raise NotImplementedError("Compact arrays not implemented for values of type {}".format(valuetype))
    if not nullfirstdate and valuetype[0:2] == "D_":
        raise NotImplementedError("Delta coding in a compact array should also NULL code the first date in the array.")

    # A compact array has a slightly weird encoding. The array specifier is
    # first followed by a "header" defining the type of each entry.
    # Then, the length of the array *in bytes*, and then the entries strung
    # together.

    # So start with only the type.
    array =  _ber_encode_type("COMP_ARR")

    # Each entry is a struct with 3 elements. The elements are:
    # a timestamp (octet string),
    # status byte (uint8 or NULLdata),
    # value (depends on application).
    # The length of the octet strings is still encoded in the remainder of the
    # array, not here.
    # It's actually unclear how compact arrays should handle the value of NULL
    # elements; we assume they can be omitted, so an untagged NULL is just an
    # empty byte array.
    array += _ber_encode_type("STRUCT", length=3)

    array += _ber_encode_type("OCTETS")

    # Since status integer is a fixed-length type, if we can't NULL at least
    # one entry we can't NULL any of them.
    array += _ber_encode_type("NULL_DATA") if nullstatus else _ber_encode_type("UINT8")

    array += _ber_encode_type(valuetype)

    # Build the rest of the array in a different bytes object so that we can
    # later insert the length.

    nulldate = nullfirstdate
    array_contents = b''
    for timestamp, value in series.iteritems():
        try:
            value = int(value)
        except ValueError:
            # FIXME how to deal with missing values?
            value = 0

        if valuetype == 'D_UINT16':
            array_contents +=_encode_delta_struct(timestamp, value,
                                                  deltalength=16,
                                                  nulldate=nulldate,
                                                  nullstatus=nullstatus,
                                                  tagged=False)
        else:
            raise NotImplementedError("Compact arrays not implemented for values of type {}".format(valuetype))

        # NULL code all dates for the remainder of the series.
        nulldate = True

    # Encode the length (in bytes, not number of entries) of the array contents.
    array += _ber_encode_type("ARR_CONTENTS", length=len(array_contents))

    # Concatenate and return the entire compact array.
    return array + array_contents


"""
Our proposed compact delta apdu uses a struct containing:
    - a measurement struct with the first measurement
    - a compact array with all the deltas following the first measurement
"""
def _encode_compact_delta_frame(dataframe, deltalength=16):
    dataframe = dataframe.copy()

    frame =  _ber_encode_type("STRUCT", length=2)

    # First timestamp
    timestamp = dataframe.index[0]
    # First measurement using the actual meter reading
    try:
        value = int(dataframe.iloc[0]["cumsum"])
    except ValueError:
        #print("Cannot generate APDU for {} due to missing initial meter reading".format(timestamp))
        return None # Cannot generate APDU without initial meter reading

    frame += _encode_measurement_struct(timestamp, value,
                                        nulldate=False, nullstatus=False,
                                        nullvalue=False, tagged=True)

    # The rest should be put in a compact array
    deltas = dataframe.iloc[1:]["measurements"]
    if deltalength == 16:
        frame += _encode_compact_array(deltas, valuetype='D_UINT16', nullfirstdate=True, nullstatus=False)
    else:
        raise NotImplementedError("Compact APDUs only implemented for 16-bit deltas")

    return frame


"""
Min-length delta encoding doesn't do NULL encoding for the values, because it's
ambiguous whether a NULL-encoded delta means the same delta as previous, or a
delta of 0.
"""
def _encode_delta_frame(dataframe, deltalength='min'):
    if deltalength not in ['min', 8, 16, 32]:
        raise ValueError("Invalid value passed for delta length: {}".format(deltalength))

    dataframe = dataframe.copy()

    frame = _ber_encode_type("ARRAY", length=len(dataframe.index))

    # First value in a frame can never be delta-encoded, so treat separately.
    # Timestamp
    timestamp = dataframe.index[0]
    # Actual meter reading
    try:
        measurement = int(dataframe.iloc[0]["cumsum"])
    except ValueError:
        #print("Cannot generate APDU for {} due to missing initial meter reading".format(timestamp))
        return None # Cannot generate APDU without initial meter reading

    frame += _encode_measurement_struct(timestamp, measurement,
                                        nulldate=False, nullstatus=False,
                                        nullvalue=False, tagged=True)

    # Subsequent values can be delta- and null-encoded.
    for timestamp, data in dataframe.iloc[1:].iterrows():
        # Select the delta
        try:
            value = int(data["measurements"])
        except ValueError:
            # TODO different way of dealing with missing values?
            # Right now we just set the delta to 0.
            value = 0

        try:
            frame += _encode_delta_struct(timestamp, value,
                                          deltalength=deltalength,
                                          nulldate=True, nullstatus=True,
                                          tagged=True)
        except ValueError as err:
            # Frame contains integer that is not encodable.
            # Expected on 8-bit fixed-length encoding.
            if deltalength != 8:
                raise err
            else:
                #print("Skipped unencodable APDU")
                return None

    return frame


def _encode_normal_frame(dataframe, nullcoding=False):
    nulldate = False
    nullstatus = False
    nullvalue = False
    if nullcoding:
        nulldate = True
        nullstatus = True
        if nullcoding == 'all':
            nullvalue = True


    dataframe = dataframe.copy()

    frame = _ber_encode_type("ARRAY", length=len(dataframe.index))

    # First value in a frame can never be null-encoded, so treat separately.
    # Timestamp
    timestamp = dataframe.index[0]
    # Actual meter reading
    try:
        value = int(dataframe.iloc[0]["cumsum"])
    except ValueError:
        #print("Cannot generate APDU for {} due to missing initial meter reading".format(timestamp))
        return None # Cannot generate APDU without initial meter reading

    frame += _encode_measurement_struct(timestamp, value,
                                        nulldate=False, nullstatus=False,
                                        nullvalue=False, tagged=True)

    # Subsequent values can be null-encoded.
    for timestamp, data in dataframe.iloc[1:].iterrows():
        prev = value

        try:
            value = int(data["cumsum"])
        except ValueError:
            pass # Just reuse the old meter data.

        if value == prev and nullvalue:
            # NULL-encode the value
            frame += _encode_measurement_struct(timestamp, None,
                                                nulldate=nulldate, nullstatus=nullstatus,
                                                nullvalue=nullvalue, tagged=True)
        else:
            # Encode the value as the smallest integer possible
            frame += _encode_measurement_struct(timestamp, value,
                                                nulldate=nulldate, nullstatus=nullstatus,
                                                nullvalue=nullvalue, tagged=True)

    return frame


def generate_apdu(dataframe, nullcoding=False, deltacoding=False, compactarray=False):
    if nullcoding and (nullcoding not in ['dates', 'all']):
        raise ValueError("Invalid value passed for null encoding: {}".format(nullcoding))
    if deltacoding and (deltacoding not in ['min', 8, 16, 32]):
        raise ValueError("Invalid value passed for delta encoding: {}".format(deltacoding))
    if compactarray:
        if not deltacoding:
            raise NotImplementedError("Compact arrays for normal encoding not implemented.")
        elif deltacoding == 'min':
            raise NotImplementedError("Compact arrays and minimum-length delta encoding cannot be combined.")

    # All APDUs have the same header
    header = b'\xc4\x01\x00\x00'

    if not deltacoding:
        apdu = _encode_normal_frame(dataframe, nullcoding=nullcoding)
    elif not compactarray:
        apdu = _encode_delta_frame(dataframe, deltalength=deltacoding)
    else:
        apdu = _encode_compact_delta_frame(dataframe, deltalength=deltacoding)

    if apdu:
        apdu = header + apdu

    return apdu

